using System;
using System.Xml;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// This class represents security response message that has optional 
	/// informative message node embedded into response element. 
	/// </summary>
	public abstract class SecurityBaseCommentedResponseMessage : SecurityBaseResponseMessage
	{
		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_MESSAGE_NODE_NAME = "Message";

		private string _message = null; // optional message

		/// <summary>
		/// This constructor allows creating security base commented response message with full 
		/// set of parameters including optional informative message. 
		/// </summary>
		/// <param name="xsdName">VistALink message XSD schema file name.</param>
		/// <param name="responseType">VistALink message response type.</param>
		/// <param name="responseStatus">VistALink message response status.</param>
		/// <param name="message">Informative message. Optional.</param>
		protected SecurityBaseCommentedResponseMessage( string xsdName, string responseType, ResponseStatus responseStatus, string message )
			: base( xsdName, responseType, responseStatus )
		{
			_message = message;
		}

		/// <summary>
		/// Standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="xsdName">VistALink message XSD schema file name.</param>
		/// <param name="responseType">VistALink message response type.</param>
		/// <param name="sourceDoc">Source XML document.</param>
		protected SecurityBaseCommentedResponseMessage( string xsdName, string responseType, XmlDocument sourceDoc )
			: base( xsdName, responseType, sourceDoc )
		{
			XmlElement _msgNode = XmlUtility.ParseGetOptionalElementByUniqueTagName( sourceDoc, XMLCONSTS_MESSAGE_NODE_NAME );
			_message = ( _msgNode == null ) ? null : _msgNode.InnerText;
		}

		///<summary>
		/// Overriden method adding content to response element 
		/// of XML message representation during XML serialization. 
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected override void WriteXmlResponseNodeContent( XmlWriter writer )
		{
			if( writer == null ) 
				throw( new ArgumentNullException( "writer" ) );

			base.WriteXmlResponseNodeContent( writer );

			if( _message != null )
				writer.WriteElementString( XMLCONSTS_MESSAGE_NODE_NAME, _message );
		}

		/// <summary>
		/// Optional informative message. 
		/// </summary>
		public string Message
		{
			get
			{
				return _message;
			}
		}
	}
}
